bl_info = {
    "name": "LPTK: LowPoly-ToolKit",
    "author": "Joshua Battenfeld",
    "version": (1, 0),
    "blender": (4, 3, 0),
    "location": "View3D > Sidebar > LPTK",
    "description": "Many GeometryNode-Setups doing all kinds of things",
    "category": "Object",
}

import bpy
import os
import bpy.utils.previews
import json

from . import exporter
from . import meshrenamer

# Get the directory of the current script (your __init__.py)
addon_dir = os.path.dirname(os.path.abspath(__file__))

# Path to 'my_geonodes' abd "thumbnails" folder
geonodes_folder = os.path.join(addon_dir, "my_geonodes")
thumbnails_folder = os.path.join(addon_dir, "thumbnails")

# --- Auto-install Asset Pack ---
from . import assetpack_installer
assetpack_installer.register_asset_library(addon_dir)

# Load geo_nodes from JSON
json_path = os.path.join(addon_dir, "geo_nodes.json")

with open(json_path, "r") as f:
    raw_nodes = json.load(f)

geo_nodes = {}

for name, data in raw_nodes.items():
    geo_nodes[name] = {
        "filepath": os.path.join(geonodes_folder, data["filename"]),
        "node_type": data["node_type"],
        "category": data["category"],
        "thumbnail": data["thumbnail"]
            if os.path.isabs(data["thumbnail"])
            else os.path.join(thumbnails_folder, data["thumbnail"])
    }

def get_thumbnail_path(thumbnail_name):
    return os.path.join(thumbnails_folder, thumbnail_name)

class GEO_OT_spawn(bpy.types.Operator):
    bl_idname = "geo.spawn"
    bl_label = "Spawn Geo Node"
    bl_description = "Spawn the selected geometry node asset"
    bl_options = {'REGISTER', 'UNDO'}

    node_group_name: bpy.props.StringProperty()
    filepath: bpy.props.StringProperty()
    node_type: bpy.props.StringProperty()

    def execute(self, context):
        self.report({'INFO'}, f"Spawning {self.node_group_name}...")
        try:
            with bpy.data.libraries.load(self.filepath, link=False) as (data_from, data_to):
                if self.node_group_name in data_from.node_groups:
                    data_to.node_groups.append(self.node_group_name)
                else:
                    self.report({'ERROR'}, f"Node group '{self.node_group_name}' not found in {self.filepath}")
                    return {'CANCELLED'}

            if self.node_type == "CURVE_LOW":
                bpy.ops.curve.primitive_bezier_curve_add(enter_editmode=True, location=context.scene.cursor.location)
                obj = bpy.context.active_object
                obj.data.resolution_u = 3
                bpy.ops.curve.select_all(action='SELECT')
                bpy.ops.curve.delete(type='VERT')
                bpy.ops.wm.tool_set_by_id(name="builtin.draw")
                settings = context.scene.tool_settings.curve_paint_settings
                settings.depth_mode = 'SURFACE'
                settings.use_stroke_endpoints = True
            elif self.node_type == "PLANE":
                bpy.ops.mesh.primitive_plane_add(size=10, location=context.scene.cursor.location)
            elif self.node_type == "TERRAIN2":
                bpy.ops.mesh.primitive_plane_add(size=50, location=context.scene.cursor.location)
                bpy.ops.object.mode_set(mode='EDIT')
                bpy.ops.mesh.subdivide(number_cuts=3)
                bpy.ops.object.mode_set(mode='OBJECT')
            elif self.node_type == "CUBE":
                bpy.ops.mesh.primitive_cube_add(enter_editmode=True, location=context.scene.cursor.location)
                obj = bpy.context.active_object
                bpy.ops.object.mode_set(mode='EDIT')
                bpy.ops.mesh.select_all(action='SELECT')
                bpy.ops.wm.tool_set_by_id(name="builtin.add_primitive_cube")
            elif self.node_type == "GATE":
                # Add cube at cursor location
                bpy.ops.mesh.primitive_cube_add(enter_editmode=True, location=context.scene.cursor.location)
                obj = bpy.context.active_object

                # Get or create "Gates" collection
                collection = bpy.data.collections.get("Gates")
                if not collection:
                    collection = bpy.data.collections.new("Gates")
                    bpy.context.scene.collection.children.link(collection)

                # Link the object to the "Gates" collection
                if obj.name not in collection.objects:
                    collection.objects.link(obj)

                # Optional: Unlink from the default collection (to move instead of duplicate)
                if obj.name in bpy.context.scene.collection.objects:
                    bpy.context.scene.collection.objects.unlink(obj)
            elif self.node_type == "CURVE":
                bpy.ops.curve.primitive_bezier_curve_add(enter_editmode=True, location=context.scene.cursor.location)
                obj = bpy.context.active_object
                obj.data.resolution_u = 12
                bpy.ops.curve.select_all(action='SELECT')
                bpy.ops.curve.delete(type='VERT')
                bpy.ops.wm.tool_set_by_id(name="builtin.draw")
                settings = context.scene.tool_settings.curve_paint_settings
                settings.depth_mode = 'SURFACE'
                settings.use_stroke_endpoints = False
            elif self.node_type == "SCATTER":
                # Store the currently active object before adding new hair
                original_active_obj = context.active_object

                # Add empty hair. This creates a *new* 'Curves' object and makes it active.
                # Use the cursor location as you do for other new object types
                bpy.ops.object.curves_empty_hair_add(align='WORLD', location=context.scene.cursor.location, scale=(1, 1, 1))

                # The newly created 'Curves' object for hair is now the active object
                hair_obj = context.active_object

                # Remove the default Geometry Nodes modifier from the *newly created hair object*
                for modifier in hair_obj.modifiers:
                    if modifier.type == 'NODES': # Check for Geometry Nodes modifier type
                        hair_obj.modifiers.remove(modifier)
                        break # Assuming there's only one default GeoNodes modifier to remove
            else:
                self.report({'WARNING'}, f"Unknown node type: {self.node_type}. Creating an empty.")
                bpy.ops.object.empty_add(location=context.scene.cursor.location)

            obj = bpy.context.active_object
            obj.name = self.node_group_name

            modifier = obj.modifiers.new(name="GeometryNodes", type='NODES')
            if bpy.data.node_groups.get(self.node_group_name):
                modifier.node_group = bpy.data.node_groups[self.node_group_name]
            else:
                self.report({'ERROR'}, f"Appended node group '{self.node_group_name}' not found in bpy.data.node_groups.")
                bpy.data.objects.remove(obj, do_unlink=True)
                return {'CANCELLED'}

            self.report({'INFO'}, f"Successfully spawned {self.node_group_name}")
            return {'FINISHED'}

        except Exception as e:
            self.report({'ERROR'}, f"Failed to spawn {self.node_group_name}: {e}")
            return {'CANCELLED'}

preview_collections = {}

def get_categories(self, context):
    categories = set()
    for name, data in geo_nodes.items():
        categories.add((data["category"], data["category"], ""))
    # Add an "All" option
    categories.add(("All", "All Categories", "Show assets from all categories"))
    return sorted(list(categories))

# Main Panel (Container)
class GEO_PT_panel(bpy.types.Panel):
    bl_label = "LPTK"
    bl_idname = "GEO_PT_panel"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI' 
    bl_category = "LPTK"

    def draw_header(self, context):
        layout = self.layout
        layout.label(text="", icon='NODETREE')

    def draw(self, context):
        # Leave empty, sub-panels handle content
        pass


# Assets (Category Filter + Node Assets in one sub-panel)
class GEO_PT_panel_assets(bpy.types.Panel):
    bl_label = "Assets"
    bl_idname = "GEO_PT_panel_assets"
    bl_parent_id = "GEO_PT_panel"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "LPTK"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        scene = context.scene

        # --- Category Filter ---
        if hasattr(scene, "geo_spawner_category"):
            layout.prop(scene, "geo_spawner_category")
            selected_category = scene.geo_spawner_category
        else:
            selected_category = "All"

        # --- Node Assets ---
        box = layout.box()
        col = box.column(align=True)

        num_cols = 2
        current_col = 0
        row = col.row(align=True)

        pcoll = preview_collections.get("thumbnails")

        displayed_nodes = {
            name: data for name, data in geo_nodes.items()
            if selected_category == "All" or data["category"] == selected_category
        }

        for name, data in displayed_nodes.items():
            if current_col % num_cols == 0 and current_col != 0:
                row = col.row(align=True)

            asset_box = row.box()
            asset_col = asset_box.column(align=True)

            if pcoll and name in pcoll:
                asset_col.template_icon(icon_value=pcoll[name].icon_id, scale=5)
            else:
                asset_col.label(text="No Preview")

            op = asset_col.operator("geo.spawn", text=f"{name}")
            op.node_group_name = name
            op.filepath = data["filepath"]
            op.node_type = data["node_type"]

            current_col += 1


# Exporter Subpanel
class GEO_PT_panel_exporter(bpy.types.Panel):
    bl_label = "Collection Exporter"
    bl_idname = "GEO_PT_panel_exporter"
    bl_parent_id = "GEO_PT_panel"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "LPTK"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        exporter_drawing_instance = exporter.ExporterPanelDrawing()
        exporter_drawing_instance.draw(context, layout)

# MeshRenamer Subpanel
class GEO_PT_panel_meshrenamer(bpy.types.Panel):
    bl_label = "Mesh Renamer"
    bl_idname = "GEO_PT_panel_meshrenamer"
    bl_parent_id = "GEO_PT_panel"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "LPTK"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        layout.operator("geo.rename_meshes", icon='OUTLINER_OB_MESH')


# List of classes to register from this __init__.py
classes = [
    GEO_OT_spawn,
    GEO_PT_panel,
    GEO_PT_panel_assets,
    GEO_PT_panel_exporter,
    GEO_PT_panel_meshrenamer,
]

def register():
    global preview_collections

    for cls in classes:
        bpy.utils.register_class(cls)

    if not hasattr(bpy.types.Scene, "geo_spawner_category"):
        bpy.types.Scene.geo_spawner_category = bpy.props.EnumProperty(
            items=get_categories,
            name="Category",
            description="Filter assets by category",
        )

    pcoll = bpy.utils.previews.new()
    preview_collections["thumbnails"] = pcoll

    for name, data in geo_nodes.items():
        thumb_path = get_thumbnail_path(data["thumbnail"])
        if os.path.exists(thumb_path):
            pcoll.load(name, thumb_path, 'IMAGE')

    exporter.register()
    meshrenamer.register()

def unregister():
    global preview_collections

    exporter.unregister()
    meshrenamer.unregister()

    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)

    if hasattr(bpy.types.Scene, "geo_spawner_category"):
        del bpy.types.Scene.geo_spawner_category

    for pcoll in preview_collections.values():
        bpy.utils.previews.remove(pcoll)
    preview_collections.clear()

if __name__ == "__main__":
    try:
        unregister()
    except Exception as e:
        print(f"Unregister failed (first run?): {e}")
    register()
